import os, yaml
import requests
import importlib.resources as pkg_resources
from jinja2 import Environment, FileSystemLoader, select_autoescape, Template

class NebuloidBuilder:
    def __init__(self, orm, api, base_dir="pages"):
        self.base_dir = base_dir
        self.api = api
        self.orm = orm

        # Jinja environment (supports extends/includes)
        self.env = Environment(
            loader=FileSystemLoader([self.base_dir,"shared"]),
            autoescape=select_autoescape(["html", "jinja"])
        )

    async def build(self, route_name: str, user_id) -> str:
        recipe_file = os.path.join(self.base_dir, route_name, "recipe.yaml")
        if not os.path.exists(recipe_file):
            raise FileNotFoundError(f"Recipe not found for route '{route_name}'")

        with open(recipe_file, "r") as f:
            recipe_data = yaml.safe_load(f) or {}

        page_data = recipe_data.get("page", {})
        template_name = page_data.get("template")
        meta_data = recipe_data.get("data")
        
        datas = await self.get_data(meta_data['sources'], user_id) if meta_data else {}

        print("data fetched:", datas)

        if not template_name:
            raise ValueError(f"⚠ No template defined in recipe for '{route_name}'")

        # Ensure clean relative path
        template_path = os.path.normpath(os.path.join(route_name, template_name)).replace("\\", "/")

        try:
            template = self.env.get_template(template_path)
        except Exception as e:
            raise FileNotFoundError(
                f"Template not found: {template_path}\n"
                f"Jinja2 error: {e}\n"
                f"Search paths: {self.env.loader.searchpath if hasattr(self.env.loader, 'searchpath') else 'unknown'}"
            ) from e

        # Just render the template (static for now)
        return template.render(datas=datas, utils=lambda x: ('/utils_'+x), static=lambda x: (f'/static_{route_name}/'+x))
    
    async def get_data(self, sources, user_id):
        results = {}
        for source in sources:
            print("Data source:", source["name"])

            if source.get('type') == 'sql':
                query = source.get('query')
                try:
                    result = await self.orm.execute(query)  # ✅ now valid
                    results[source["name"]] = result
                except Exception as e:
                    results[source["name"]] = {
                        "status": "error",
                        "message": str(e)
                    }
            elif source.get('type') == 'rest':
                endpoint = source.get("endpoint")
                try:
                    resp = requests.get(endpoint, timeout=10)
                    resp.raise_for_status()
                    results[source["name"]] = {
                        "status": "success",
                        "data": resp.json()
                    }
                except Exception as e:
                    results[source["name"]] = {
                        "status": "error",
                        "message": str(e)
                    }
            elif source.get('type') == 'internal':
                print("Source", source)
                req_datas = source.get("datas", [])
                internal_results = {}
                for req in req_datas:
                    if req == "profile":
                        internal_results["profile"] = await self.orm.get_user_profile(user_id)
                    elif req == "settings":
                        internal_results["settings"] = await self.orm.execute("SELECT preferences FROM testdb.users where id= :user_id", {"user_id": user_id})
                results[source["name"]] = internal_results
            elif source.get('type') == 'portal':
                api_data, code = await self.api.handle_raw("data", {"name": source.get("portal_name", ""), "args": source.get("args", {})}, user_id=user_id)
                results[source["name"]] = api_data['result'] if code == 200 else {"status": "error", "message": "Portal API error"}
            else:
                results[source["name"]] = {"status": "error", "message": "Unknown source type"}

        return results
    def gen_utils(self, info):
            if info['name'] == 'portal.js':
                func_names = info['args']['func_names']

                # read template content as string
                template_path = pkg_resources.files('nebuloid.utils').joinpath('template/portal.js.jinja')
                template_content = template_path.read_text()  # reads as str

                # render template
                template = Template(template_content)
                output = template.render(func_names=func_names)

                # write to shared folder
                output_path = os.path.join("shared", "portal.js")
                with open(output_path, "w") as f:
                    f.write(output)

                print(f"Generated {output_path}")
            else:
                print(f"Unknown utility: {info['name']}")